"""Test Wavelet packet Frechet distance."""
import pytest
import torch as th
import pytest
import numpy as np

from itertools import pairwise
from src.fwd import compute_packet_statistics, compute_avg_frechet_distance
from sklearn.datasets import load_sample_images
from torchvision import transforms
from copy import deepcopy

th.set_default_dtype(th.float64)
th.use_deterministic_algorithms(True)

make_dataloader = lambda x: th.utils.data.DataLoader(th.utils.data.TensorDataset(x), batch_size=1, shuffle=False, drop_last=False)
default_params = {
    'dataloader': None,
    'wavelet': 'Haar',
    'max_level': 1,
    'log_scale': False
}

@pytest.mark.slow
def get_images(img_size: int = 32) -> th.Tensor:
    """Generate images of given size.

    Args:
        img_size (int): Integer specifying the desired image size.

    Returns:
        th.Tensor: Tensor containing images of shape [batchsize, channels, height, width].

    """
    dataset = load_sample_images()
    tower = th.Tensor(deepcopy(dataset.images[0]))
    flower = th.Tensor(deepcopy(dataset.images[1]))
    images = th.stack([tower, tower, tower, tower, flower, flower, flower, flower], axis=0)
    images = images/255.0
    images = th.permute(images, [0, 3, 1, 2])
    images = transforms.functional.resize(images, (img_size, img_size))
    assert images.shape == (8, 3, img_size, img_size)
    return images


def compute_fwd(target_images, output_images):
    default_params['dataloader'] = make_dataloader(target_images)
    mu1, sigma1 = compute_packet_statistics(**default_params)
    default_params['dataloader'] = make_dataloader(output_images)
    mu2, sigma2 = compute_packet_statistics(**default_params)
    
    distance = compute_avg_frechet_distance(mu1, mu2, sigma1, sigma2)
    return distance


@pytest.mark.slow
def test_same_input():
    target_images = get_images()
    output_images = deepcopy(target_images)
    fwd = compute_fwd(target_images, output_images)
    assert np.allclose(fwd, 0.0, atol=1e-3)


@pytest.mark.slow
def test_shuffle_input():
    target_images = get_images()
    # Shuffle the output images
    output_images = deepcopy(target_images)
    permutation = th.randperm(len(target_images))
    shuffled_images = output_images[permutation, :, :, :]
    assert not th.allclose(shuffled_images, output_images)

    shuffled_fwd = compute_fwd(target_images, shuffled_images)
    unshuffled_fwd = compute_fwd(target_images, output_images)
    assert np.allclose(shuffled_fwd, unshuffled_fwd, atol=1e-5)


@pytest.mark.slow
@pytest.mark.parametrize("img_size_level", [(32, 1), (64, 2)])
def test_various_image_sizes(img_size_level):
    size, level = img_size_level

    default_params['max_level'] = level

    target_images = get_images(size)
    output_images = deepcopy(target_images)
    assert output_images.shape == (8, 3, size, size)
    
    fwd = compute_fwd(target_images, output_images)
    assert np.allclose(fwd, 0.0, atol=1e-3)

